/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Playback/EaseCurves/EaseCurvesDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>

const int GRAPH_CANVAS_HEIGHT=64;
const int GRAPH_CANVAS_WIDTH=64;

EaseCurvesDemo::EaseCurvesDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{

	// Disable warnings: if no renderer									
	if( hkString::strCmp( m_env->m_options->m_renderer, "n" ) == 0 )
	{
		setErrorEnabled(0xf0d1e423, false); //'Could not realize an inplace texture of type PNG.'
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from( 2,-2,1 );
		hkVector4 to  ( 0,0,0 );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	m_loader = new hkLoader();

	// Convert the scene
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene_Default.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

		HK_ASSERT2(0x27343635, scene, "No scene loaded");
		removeLights(m_env);
		env->m_sceneConverter->convert( scene );
	}

	// Get the rig
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];
	}

	// Get the animation and the binding
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRunLoop_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRunLoop.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		m_animation[0] =  ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[0] = ac->m_bindings[0];

#if defined(HK_REAL_IS_DOUBLE)
		assetFile = "Resources/Animation/HavokGirl/hkWaveLoop_DP.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#else
		assetFile = "Resources/Animation/HavokGirl/hkWaveLoop.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#endif
		container = m_loader->load( assetFile.cString() );
		ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		m_animation[1] =  ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[1] = ac->m_bindings[0];

#if defined(HK_REAL_IS_DOUBLE)
		assetFile = "Resources/Animation/HavokGirl/hkIdle_DP.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#else
		assetFile = "Resources/Animation/HavokGirl/hkIdle.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#endif
		container = m_loader->load( assetFile.cString() );
		ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		m_animation[2] =  ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[2] = ac->m_bindings[0];
	}

	// Convert the skin
	{
#if defined(HK_REAL_IS_DOUBLE)
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin_DP.hkt";
#else
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin.hkt";
#endif
		hkStringBuf assetFile(skinFile); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );
	}

	// Create the skeleton
	m_skeletonInstance = new hkaAnimatedSkeleton( m_skeleton );
	m_skeletonInstance->setReferencePoseWeightThreshold( 0.1f );

	// Grab the animations
	for (int i=0; i < NUM_ANIMS; i++)
	{
		const hkBool startEasedIn = ( i == 2 ); // Start with hands on hips only
		m_control[i] = new hkaDefaultAnimationControl( m_binding[i], startEasedIn );
		m_control[i]->setMasterWeight( 1.0f );
		m_control[i]->setPlaybackSpeed( 1.0f );

		m_skeletonInstance->addAnimationControl( m_control[i] );
		m_control[i]->removeReference();
	}

	// Set ease curves explicitly
	m_control[0]->setEaseInCurve(0, 0, 1, 1);	// Smooth
	m_control[0]->setEaseOutCurve(1, 1, 0, 0);	// Smooth

	m_control[1]->setEaseInCurve(0, 0.33f, 0.66f, 1);	// Linear
	m_control[1]->setEaseOutCurve(1, 0.66f, 0.33f, 0);	// Linear

	m_control[2]->setEaseInCurve(0, 0, 0, 1);	// Fast
	m_control[2]->setEaseOutCurve(1, 0, 0, 0);	// Fast

	// make a world so that we can auto create a display world to hold the skin
	setupGraphics( );

	//
	//	Create the texture
	//
	{
		m_graphCanvasData = hkAllocate<unsigned char> (GRAPH_CANVAS_HEIGHT*GRAPH_CANVAS_WIDTH*4, HK_MEMORY_CLASS_DEMO);
		m_context = env->m_window->getContext();
		m_texture = hkgTexture::create( m_context );
		m_texture->allocateSurface( GRAPH_CANVAS_WIDTH, GRAPH_CANVAS_HEIGHT, true, false, m_graphCanvasData );
		m_texture->realize(true);
	}
}

EaseCurvesDemo::~EaseCurvesDemo()
{	
	m_skeletonInstance->removeReference();
	m_skinBindings.clear();

	delete m_loader;

	hkDeallocate<unsigned char>(m_graphCanvasData);

	m_env->m_window->getContext()->lock();
	m_texture->free();
	m_texture->release();
	m_env->m_window->getContext()->unlock();
}

const hkColor::Argb curveCols[3] =
{
	hkColor::YELLOW,
	hkColor::MAGENTA,
	hkColor::CYAN
};

hkDemo::Result EaseCurvesDemo::stepDemo()
{

	// Display current settings
	{
		char buf[255];
		for (int i=0; i< NUM_ANIMS; i++)
		{
			hkString::sprintf(buf, "anim%d:%0.3f", i, m_control[i]->getWeight());
			m_env->m_textDisplay->outputText( buf, 10, 240+14*i, curveCols[i], 1);
		}
	}

	// Check user input
	{
		for (int i=0; i < 3; i++)
		{
			if (m_env->m_gamePad->wasButtonPressed( 1 << (i+1) ))
			{
				if (( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASING_IN ) ||
					( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASED_IN ))
				{
					m_control[i]->easeOut( 1.0f );
				}
				else
				{
					m_control[i]->easeIn( 1.0f );
				}
			}
		}
	}

	// Advance the active animations
	m_skeletonInstance->stepDeltaTime( 1.0f / 60.0f );

	// Sample the active animations and combine into a single pose
	hkaPose pose (m_skeleton);
	m_skeletonInstance->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), pose.getFloatSlotValues().begin()  );
	AnimationUtils::drawPose( pose, hkQsTransform::getIdentity() );

	// Skin the meshes
	AnimationUtils::skinMesh( pose, m_skinBindings, hkTransform::getIdentity(), *m_env->m_sceneConverter );

	// Show curves
	{
		int* data = reinterpret_cast<int*>(m_texture->lock(0, true));

		//Shuffle Texture
		for (int w=0; w< GRAPH_CANVAS_WIDTH-1; w++)
		{
			for (int h=0; h< GRAPH_CANVAS_HEIGHT; h++)
			{
				data[h * GRAPH_CANVAS_WIDTH + w] = data[h * GRAPH_CANVAS_WIDTH + w+1];
			}
		}

		// Clear the line
		for (int h=0; h< GRAPH_CANVAS_HEIGHT; h++)
		{
			data[h * GRAPH_CANVAS_WIDTH + GRAPH_CANVAS_WIDTH-1] = 0xff7f7f7f;
		}

		// Take a new sample
		for (int c=0; c< 3; c++)
		{
			int y = (int)(m_control[c]->getWeight() * (GRAPH_CANVAS_HEIGHT-5));
			data[(GRAPH_CANVAS_HEIGHT-2 - (y+c+1)) * GRAPH_CANVAS_WIDTH + GRAPH_CANVAS_WIDTH-1] = curveCols[c];
			data[(GRAPH_CANVAS_HEIGHT-2 - (y+c)) * GRAPH_CANVAS_WIDTH + GRAPH_CANVAS_WIDTH-1] &= 0xff0f0f0f;
		}

		m_texture->unlock(0);
	}

 	m_context->lock();

 	drawTexture( m_texture );

 	m_context->unlock();

	return hkDemo::DEMO_OK;
}


void EaseCurvesDemo::drawTexture( hkgTexture* texture )
{
	hkgViewport* orthoView = m_env->m_window->getWindowOrthoView();  //2D ortho, width by height
	float windowHeight = (float)getWindowHeight();

	m_context->lock();

	orthoView->setAsCurrent(m_context); // 2D

	m_context->setDepthReadState(false);
	m_context->setDepthWriteState(true);

	m_context->setCurrentSoleTexture( texture, HKG_TEXTURE_MODULATE );
	m_context->setTexture2DState( true ); // turn on textures for this
	m_context->setBlendState( false );

	float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f};
	m_context->setCurrentColor4( white );

	float p[3];
	float uv[2];
	p[2] = -0.01f; //depth (0..-2)

	float tl[3];
	float lr[3];

	float wOffset = 120;
	float hOffset = 230;

	tl[0] = 0.0f + wOffset;
	tl[1] = windowHeight  - hOffset;

	const int TEXTURE_SCALE = 2;
	lr[0] = float( GRAPH_CANVAS_WIDTH * TEXTURE_SCALE) + wOffset;
	lr[1] = windowHeight - float( GRAPH_CANVAS_HEIGHT) - hOffset;

	m_context->beginGroup( HKG_IMM_TRIANGLE_LIST );

	p[0] = tl[0]; p[1] = tl[1];
	uv[0] = 0.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = lr[1];
	uv[0] = 1.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	m_context->endGroup();

	m_context->setDepthReadState(true);
	m_context->setDepthWriteState(true);
	m_context->setCurrentSoleTexture( HK_NULL, HKG_TEXTURE_MODULATE );

	m_context->unlock();
}


HK_DECLARE_DEMO(EaseCurvesDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Ease Curves", "Press \x11,\x12,\x13 to ease in / out different animations");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
