/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Multithreading/SkeletonMapping/SkeletonMappingMultithreadingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Mapper/Multithreaded/hkaAnimationMappingJobQueueUtils.h>
#include <Animation/Animation/Playback/Multithreaded/SampleAndCombine/hkaAnimationSampleAndCombineJobQueueUtils.h>
#include <Animation/Animation/Playback/Multithreaded/SampleAndCombine/hkaAnimationSampleAndCombineJobs.h>
#include <Animation/Animation/Playback/Multithreaded/hkaMultithreadedAnimationUtils.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/WindowedAverage/WindowedAverage.h>
#include <Animation/Animation/Rig/hkaSkeletonUtils.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Common/Base/Types/Color/hkColor.h>


static const int NUM_CHARACTERS = 10;

struct SkeletonMappingMultithreadingVariant
{
	const char*	m_name;
	const char* m_details;
	const char* m_rigFileNameA;
	const char* m_rigFileNameB;
	const char* m_animationFileName;
	hkBool m_ragdollMapping;
	hkReal m_scale;
};

static const SkeletonMappingMultithreadingVariant g_variants[] =
{
	{
		"Ragdoll Mapping",
		"Press \x11 to toggle multithreaded mode",
		"Resources/Animation/Mapping/bipedHighRig.hkt",
		"Resources/Animation/Mapping/bipedLowRig.hkt",
		"Resources/Animation/Mapping/bipedHighAnim.hkt",
		true,
		250.0f
	},
	{
		"Animation Retargeting",
		"Press \x11 to toggle multithreaded mode",
		"Resources/Animation/Retargeting/shulgoth_default.hkt",
		"Resources/Animation/Retargeting/male_nocape.hkt",
		"Resources/Animation/Retargeting/shulgoth_default_walk.hkt",
		false,
		20.0f
	}
};


static int HK_CALL hkRemoveHighLowPrefixFilter(const char* str1, const char* str2)
{
	hkStringBuf string1(str1);
	hkStringBuf string2(str2);

	if (string1.startsWith("High "))
	{
		string1.chompStart(5);
	}
	if (string2.startsWith("High "))
	{
		string2.chompStart(5);
	}

	if (string1.startsWith("Low "))
	{
		string1.chompStart(4);
	}
	if (string2.startsWith("Low "))
	{
		string2.chompStart(4);
	}

	return string1.compareToIgnoreCase( string2 );
}

SkeletonMappingMultithreadingDemo::SkeletonMappingMultithreadingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	m_useMt = true;
#else
	m_useMt = false;
#endif

	const SkeletonMappingMultithreadingVariant &variant = g_variants[ m_variantId ];

	m_loader = new hkLoader();

	//
	// Setup the camera
	//
	{
		hkVector4 from(  0.0f, -variant.m_scale,  0.0f);
		hkVector4 to  (  0.0f,  0.0f,   0.0f);
		hkVector4 up  (  0.0f,  0.0f,   1.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Skeletons
	{

		{
			hkStringBuf assetFile( variant.m_rigFileNameA );
			hkAssetManagementUtil::getFilePath( assetFile );
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

			HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load high-res skeleton");
			m_highSkeleton = ac->m_skeletons[0];

		}

		{
			hkStringBuf assetFile( variant.m_rigFileNameB );
			hkAssetManagementUtil::getFilePath( assetFile );
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

			HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load low-res skeleton");
			m_lowSkeleton = ac->m_skeletons[0];

		}
	}

	// Animations
	{
		// High res animation
		hkStringBuf assetFile( variant.m_animationFileName );
		hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "Couldn't load high-res animation");
		m_highAnimation = ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "Couldn't load high-res binding");
		m_highBinding = ac->m_bindings[0];

		// Create several animated skeletons
		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			// Create an animation control
			hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_highBinding );

			// Create a new animated skeleton
			hkaAnimatedSkeleton* skeleton = new hkaAnimatedSkeleton( m_highSkeleton );

			// Bind the control to the skeleton
			skeleton->addAnimationControl( control );

			// The animated skeleton now owns the control
			control->removeReference();

			// Offset in time
			skeleton->stepDeltaTime( i * 0.25f );

			m_highAnimatedSkeletons.pushBack( skeleton );
		}
	}

	// Poses
	{
		m_highPoses.setSize( NUM_CHARACTERS );
		m_lowPoses.setSize( NUM_CHARACTERS );

		for ( int i = 0; i < m_highAnimatedSkeletons.getSize(); ++i )
		{
			m_highPoses[ i ] = new hkaPose( m_highSkeleton );
			m_lowPoses[ i ] = new hkaPose( m_lowSkeleton );
		}
	}

	// Add some extra constraint data on the bones
	// The only two bones that can move are the root and the pelvis
	if ( variant.m_ragdollMapping )
	{
		{
			hkaSkeletonUtils::lockTranslations( *m_highSkeleton );

			const hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(*m_lowSkeleton, "Low Pelvis");
			m_lowSkeleton->m_bones[pelvis].m_lockTranslation = false;
		}

		{
			hkaSkeletonUtils::lockTranslations( *m_lowSkeleton );

			const hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(*m_highSkeleton, "High Pelvis");
			m_highSkeleton->m_bones[pelvis].m_lockTranslation = false;
		}
	}

	setupGraphics( );

	hkaAnimationMappingJobQueueUtils::registerWithJobQueue( m_jobQueue );
	hkaAnimationSampleAndCombineJobQueueUtils::registerWithJobQueue( m_jobQueue );


	// Create mappers
	hkaSkeletonMapperData high_low_data;
	hkaSkeletonMapperData low_high_data;
	{
		hkaSkeletonMapperUtils::Params params;

		params.m_skeletonA = m_highSkeleton;
		params.m_skeletonB = m_lowSkeleton;
		params.m_mappingType = variant.m_ragdollMapping ? hkaSkeletonMapperData::HK_RAGDOLL_MAPPING : hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
		params.m_compareNames = variant.m_ragdollMapping ? hkRemoveHighLowPrefixFilter : HK_NULL;
		params.m_positionMatchTolerance = 0.001f;
		params.m_rotationMatchTolerance = 0.001f;
		params.m_autodetectChains = variant.m_ragdollMapping;
		params.m_autodetectSimple = true;

		// We enforce a chain from the start of the spine to the neck
		if ( variant.m_ragdollMapping )
		{
			hkaSkeletonMapperUtils::UserChain chain;
			chain.m_start = "Pelvis";
			chain.m_end = "Neck";
			params.m_userChains.pushBack( chain );
		}

		hkaSkeletonMapperUtils::createMapping( params, high_low_data, low_high_data );

		low_high_data.m_keepUnmappedLocal = false;
		high_low_data.m_keepUnmappedLocal = false;
	}

	m_highToLowMapper = new hkaSkeletonMapper( high_low_data );
	m_lowToHighMapper = new hkaSkeletonMapper( low_high_data );
}


SkeletonMappingMultithreadingDemo::~SkeletonMappingMultithreadingDemo()
{
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		m_highAnimatedSkeletons[ i ]->removeReference();

		delete m_highPoses[ i ];
		delete m_lowPoses[ i ];
	}

	m_highToLowMapper->removeReference();
	m_lowToHighMapper->removeReference();
	delete m_loader;
}

hkDemo::Result SkeletonMappingMultithreadingDemo::stepDemo()
{
	const SkeletonMappingMultithreadingVariant &variant = g_variants[ m_variantId ];

	const int n = NUM_CHARACTERS;

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) )
	{
		m_useMt = !m_useMt;
	}
#endif

	// Advance the animation of each of the hi-res skeletons
	for ( int i = 0; i < m_highAnimatedSkeletons.getSize(); ++i )
	{
		m_highAnimatedSkeletons[ i ]->stepDeltaTime( m_timestep );
	}

	// Fill the high resolution poses
	if ( m_useMt )
	{
		// Sample multithreaded and wait for completion
		doSamplingMultithreaded();

		// Map multithreaded and wait for completion
		doMappingMultithreaded();
	}
	else
	{
		// Sample the high res animations
		for ( int i = 0; i < n; ++i )
		{
			m_highAnimatedSkeletons[ i ]->sampleAndCombineAnimations( m_highPoses[ i ]->accessUnsyncedPoseLocalSpace().begin(), m_highPoses[ i ]->getFloatSlotValues().begin() );
		}

		// Map to low resolution
		for ( int i = 0; i < n; ++i )
		{
			m_lowPoses[ i ]->setToReferencePose();

			// Map the high-res pos into the low-res pose
			m_highToLowMapper->mapPose( *m_highPoses[ i ], *m_lowPoses[ i ], hkaSkeletonMapper::CURRENT_POSE );
		}
	}

	// Draw both poses for each character
	{
		hkQsTransform worldFromModel( hkQsTransform::IDENTITY );

		// Draw the hi-res original poses
		for ( int i = 0; i < n; ++i )
		{
			worldFromModel.m_translation.set( ( hkReal( i ) / hkReal( n - 1 ) - 0.5f ) * variant.m_scale, 0, variant.m_scale / 5.0f );
			AnimationUtils::drawPose( *m_highPoses[ i ], worldFromModel, hkColor::GREEN );
		}

		// Draw the low-res mapped poses
		for ( int i = 0; i < n; ++i )
		{
			worldFromModel.m_translation.set( ( hkReal( i ) / hkReal( n - 1 ) -0.5f ) * variant.m_scale, 0, -variant.m_scale / 5.0f );
			AnimationUtils::drawPose( *m_lowPoses[ i ], worldFromModel, hkColor::BLUE );
		}
	}

	// Display info
	{
		hkStringBuf str;
		str += variant.m_ragdollMapping ? "Ragdoll Mapping\n" : "Animation Retargeting\n";
		str += m_useMt ? "Multi threaded" : "Single threaded";
		m_env->m_textDisplay->outputText( str.cString(), 20, (int)getWindowHeight() - 100 );
	}

	return hkDefaultAnimationDemo::stepDemo();
}

void SkeletonMappingMultithreadingDemo::doMappingMultithreaded()
{
	const SkeletonMappingMultithreadingVariant &variant = g_variants[ m_variantId ];

	// Set up the jobs
	hkLocalArray< hkaAnimationMapPoseJob > mappingJobs( NUM_CHARACTERS );
	mappingJobs.setSize( NUM_CHARACTERS );

	HK_TIMER_BEGIN( "MappingJobSetup", HK_NULL );

	// Make jobs
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		// It is important for both retargeting and ragdoll mapping that the B pose be initialized before mapping
		// as not all bones may be mapped
		m_lowPoses[ i ]->setToReferencePose();

		// This semaphore is released when the job is complete (if it's non-null)
		mappingJobs[ i ].m_jobDoneSemaphore = HK_NULL;

		mappingJobs[ i ].m_mapper = m_highToLowMapper;
		mappingJobs[ i ].m_poseBLocalSpaceIn = m_lowPoses[ i ]->getSyncedPoseLocalSpace().begin();
		mappingJobs[ i ].m_source = hkaSkeletonMapper::CURRENT_POSE;
		mappingJobs[ i ].m_blendHint = hkaAnimationBinding::NORMAL;

		if ( variant.m_ragdollMapping )
		{
			mappingJobs[ i ].m_poseAIn = m_highPoses[ i ]->getSyncedPoseModelSpace().begin();
			mappingJobs[ i ].m_poseBInOut = m_lowPoses[ i ]->accessSyncedPoseModelSpace().begin();
		}
		else
		{
			mappingJobs[ i ].m_poseAIn = m_highPoses[ i ]->getSyncedPoseLocalSpace().begin();
			mappingJobs[ i ].m_poseBInOut = m_lowPoses[ i ]->accessSyncedPoseLocalSpace().begin();
		}

		// Add the job - execution begins immediately on the SPU.
		// Since however this uses a critical section, and the job set up code above is relatively fast,
		// we defer adding until all jobs are set up, and then use an addJobBatch - this will be faster.
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN( "MappingAddJobBatch", HK_NULL );
		hkLocalArray<hkJob*> jobPointers( NUM_CHARACTERS );
		jobPointers.setSize( NUM_CHARACTERS );

		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			jobPointers[i] = &( mappingJobs[ i ] );
		}

		// Add the job - execution begins immediately on the SPU
		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	HK_TIMER_END();

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();
}


void SkeletonMappingMultithreadingDemo::doSamplingMultithreaded()
{
	// Set up the jobs
	hkLocalArray< hkaAnimationSampleAndCombineJob > sampleAndCombineJobs( NUM_CHARACTERS );
	sampleAndCombineJobs.setSize( NUM_CHARACTERS );

	HK_TIMER_BEGIN( "SamplingJobSetup", HK_NULL );

	// Make jobs
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		// We gather data for each active control and place it in a buffer to be used by the SPU
		// The buffer must persist in memory as long as the job is active.

		// Allocate space for the buffer (typically 50 bytes per active control)
		hkaMultithreadedAnimationUtils::allocateSampleAndCombineJob( m_highAnimatedSkeletons[i], sampleAndCombineJobs[i] );

		hkaMultithreadedAnimationUtils::createSampleAndCombineJob( m_highAnimatedSkeletons[i],
			m_highAnimatedSkeletons[i]->getSkeleton()->m_bones.getSize(), m_highPoses[i]->accessUnsyncedPoseModelSpace().begin(),
			m_highAnimatedSkeletons[i]->getSkeleton()->m_floatSlots.getSize(), m_highPoses[i]->getFloatSlotValues().begin(),
			const_cast< hkInt16* >( m_highSkeleton->m_parentIndices.begin() ),
			sampleAndCombineJobs[i] );


		// Add the job - execution begins immediately on the SPU.
		// However, since this uses a critical section and the job set up code above is relatively fast,
		// we defer adding until all jobs are set up, and then use an addJobBatch - this will be faster.
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN( "SamplingAddJobBatch", HK_NULL );
		hkLocalArray<hkJob*> jobPointers( NUM_CHARACTERS );
		jobPointers.setSize( NUM_CHARACTERS);

		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			jobPointers[i] = &( sampleAndCombineJobs[i] );
		}

		// Add the job - execution begins immediately on the SPU
		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	HK_TIMER_END();

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();

	// Clean up the control buffers
	for (int i = 0; i < NUM_CHARACTERS; ++i )
	{
		hkaMultithreadedAnimationUtils::deallocateSampleAndCombineJob( sampleAndCombineJobs[i] );
	}
}





static const char* description = "Demo showing animation skeleton mapping using multithreaded jobs";

#if defined(HK_REAL_IS_DOUBLE)
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( SkeletonMappingMultithreadingDemo, HK_DEMO_TYPE_ANIMATION , SkeletonMappingMultithreadingVariant, g_variants, description );
#else
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( SkeletonMappingMultithreadingDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL , SkeletonMappingMultithreadingVariant, g_variants, description );
#endif

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
