/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/MotionExtraction/MirroredMotion/MirroredMotionDemo.h>
#include <Animation/Animation/Animation/Mirrored/hkaMirroredAnimation.h>
#include <Animation/Animation/Animation/Mirrored/hkaMirroredSkeleton.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>

const int GRAPH_CANVAS_HEIGHT=64;
const int GRAPH_CANVAS_WIDTH=64;


struct MirroredMotionVariant
{
	const char* m_name;
	const char* m_animationFileName;
	const char* m_details;
};

const char *myHelpString = "\x11 Toggle Animation - \x12 Wireframe - \x13 Show Skeleton\n\x10 Toggle extracted motion - \x1B Reset extracted motion - \x1C Pause";

static const MirroredMotionVariant g_variants[] =
{
#if defined(HK_REAL_IS_DOUBLE)
	{ "Havok Girl Idle", "Resources/Animation/HavokGirl/hkIdle_DP.hkt", myHelpString },
	{ "Havok Girl Jump Land Loop", "Resources/Animation/HavokGirl/hkJumpLandLoop_DP.hkt", myHelpString },
	{ "Havok Girl Long Anim", "Resources/Animation/HavokGirl/hkLongAnim_DP.hkt", myHelpString },
	{ "Havok Girl Run Loop", "Resources/Animation/HavokGirl/hkRunLoop_DP.hkt", myHelpString },
	{ "Havok Girl Run Turn Left Loop", "Resources/Animation/HavokGirl/hkRunTurnLLoop_DP.hkt", myHelpString },
	{ "Havok Girl Run Turn Right Loop", "Resources/Animation/HavokGirl/hkRunTurnRLoop_DP.hkt", myHelpString },
	{ "Havok Girl Walk Loop", "Resources/Animation/HavokGirl/hkWalkLoop_DP.hkt", myHelpString },
	{ "Havok Girl Walk Turn Left Loop", "Resources/Animation/HavokGirl/hkWalkTurnLLoop_DP.hkt", myHelpString },
	{ "Havok Girl Walk Turn Right Loop", "Resources/Animation/HavokGirl/hkWalkTurnRLoop_DP.hkt", myHelpString },
	{ "Havok Girl Wave", "Resources/Animation/HavokGirl/hkWaveLoop_DP.hkt", myHelpString }
#else
	{ "Havok Girl Idle", "Resources/Animation/HavokGirl/hkIdle.hkt", myHelpString },
	{ "Havok Girl Jump Land Loop", "Resources/Animation/HavokGirl/hkJumpLandLoop.hkt", myHelpString },
	{ "Havok Girl Long Anim", "Resources/Animation/HavokGirl/hkLongAnim.hkt", myHelpString },
	{ "Havok Girl Run Loop", "Resources/Animation/HavokGirl/hkRunLoop.hkt", myHelpString },
	{ "Havok Girl Run Turn Left Loop", "Resources/Animation/HavokGirl/hkRunTurnLLoop.hkt", myHelpString },
	{ "Havok Girl Run Turn Right Loop", "Resources/Animation/HavokGirl/hkRunTurnRLoop.hkt", myHelpString },
	{ "Havok Girl Walk Loop", "Resources/Animation/HavokGirl/hkWalkLoop.hkt", myHelpString },
	{ "Havok Girl Walk Turn Left Loop", "Resources/Animation/HavokGirl/hkWalkTurnLLoop.hkt", myHelpString },
	{ "Havok Girl Walk Turn Right Loop", "Resources/Animation/HavokGirl/hkWalkTurnRLoop.hkt", myHelpString },
	{ "Havok Girl Wave", "Resources/Animation/HavokGirl/hkWaveLoop.hkt", myHelpString }
#endif
};



MirroredMotionDemo::MirroredMotionDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Character parameters
#if defined(HK_REAL_IS_DOUBLE)
//	const char *sceneFileName = "Resources/Animation/Scene/hkScene_Default.hkt";
	const char *rigFileName = "Resources/Animation/HavokGirl/hkRig_DP.hkt";
	const char *skinFileName = "Resources/Animation/HavokGirl/hkLowResSkin_DP.hkt";
#else
	//	const char *sceneFileName = "Resources/Animation/Scene/hkScene.hkt";
	const char *rigFileName = "Resources/Animation/HavokGirl/hkRig.hkt";
	const char *skinFileName = "Resources/Animation/HavokGirl/hkLowResSkin.hkt";
#endif

	// Mirroring Parameters
	const char * leftPrefix[2] = { " L ", "EyeL" };
	const char *rightPrefix[2] = { " R ", "EyeR" };
	const int numPrefix = 2;
	const hkQuaternion mirrorAxis( 1.0f, 0.0f, 0.0f, 0.0f );

	// Show backfaces so that you can see inside of things
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	env->m_sceneConverter->setAllowTextureSharing( true );
	env->m_sceneConverter->setAllowMaterialSharing( true );

	//
	// Setup the camera
	//
	{
		hkVector4 from( 0,-6,1 );
		hkVector4 to  ( 0,0,0 );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.01f, 100 );
	}

	m_loader = new hkLoader();

	// Get the rig
	{
		hkStringBuf assetFile(rigFileName); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];
	}

	// Get the animation and the binding
	{
		hkStringBuf assetFile = g_variants[ env->m_variantId ].m_animationFileName; hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		m_animation[0] =  ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[0] = ac->m_bindings[0];
	}

	// Create the mirrored animation and binding
	{
		hkArray< hkStringPtr > ltag;
		hkArray< hkStringPtr > rtag;

		for ( int j = 0; j < numPrefix; j++ )
		{
			ltag.pushBack( ( leftPrefix  )[ j ] );
			rtag.pushBack( ( rightPrefix )[ j ] );
		}

		hkaMirroredSkeleton *mirroredSkeleton = new hkaMirroredSkeleton( m_skeleton );

		mirroredSkeleton->computeBonePairingFromNames( ltag, rtag );

		hkaMirroredAnimation *mirroredAnimation = new hkaMirroredAnimation( m_animation[0], m_binding[0], mirroredSkeleton );
		mirroredSkeleton->removeReference();

		mirroredSkeleton->setAllBoneInvariantsFromReferencePose( mirrorAxis, 0.0f );

		m_binding[1] = mirroredAnimation->createMirroredBinding();
		m_animation[1] = mirroredAnimation;
	}

	// Create the skeletons
	for ( int i = 0; i < 2; i++ )
	{
		m_skeletonInstance[i] = new hkaAnimatedSkeleton( m_skeleton );
		m_skeletonInstance[i]->setReferencePoseWeightThreshold( 0.1f );
	}

	// Convert the skin (once for each character standard and mirrored)
	for ( int i = 0; i < 2; i++ )
	{
		hkStringBuf assetFile(skinFileName); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		if ( container != HK_NULL )
		{
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
			HK_ASSERT2(0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
			HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

			m_skinBindings[i] = ac->m_skins;

			m_attachments[i] = ac->m_attachments;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );

			// Handle the attachements
			for (int a=0; a < m_attachments[i].getSize(); ++a)
			{
				hkaBoneAttachment* ba = m_attachments[i][a];
				hkgDisplayObject* hkgObject = HK_NULL;

				//Check the attachment is a mesh
				if ( hkString::strCmp(ba->m_attachment.getClass()->getName(), hkxMeshClass.getName()) == 0)
				{
					hkgObject = env->m_sceneConverter->findFirstDisplayObjectUsingMesh((hkxMesh*)ba->m_attachment.val());
					if (hkgObject)
					{
						hkgObject->setStatusFlags( hkgObject->getStatusFlags() | HKG_DISPLAY_OBJECT_DYNAMIC);
					}
				}

				m_attachmentObjects[i].pushBack(hkgObject);
			}
		}
	}

	// Grab the animations
	for ( int i = 0; i < 2; i++ )
	{
		m_control[i] = new hkaDefaultAnimationControl( m_binding[i] );
		m_control[i]->setMasterWeight( 1.0f );
		m_control[i]->setPlaybackSpeed( 1.0f );

		// Ease all controls out initially
		m_control[i]->easeIn(1.0f);

		m_skeletonInstance[i]->addAnimationControl( m_control[i] );
		m_control[i]->removeReference();

		// Set ease curves explicitly
		m_control[i]->setEaseInCurve(0, 0, 1, 1);	// Smooth
		m_control[i]->setEaseOutCurve(1, 1, 0, 0);	// Smooth

		m_accumulatedMotion[i].setIdentity();
	}

	// make a world so that we can auto create a display world to hold the skin
	setupGraphics( );

	m_axisCounter = 0;
	m_env = env;

	m_drawSkin = true;
	m_wireframe = false;
	m_timestep = 1.0f / 60.0f;
	m_useExtractedMotion = true;
	m_paused = false;
	m_separation = 1.0f;
}

MirroredMotionDemo::~MirroredMotionDemo()
{
	m_skeletonInstance[0]->removeReference();
	m_skeletonInstance[1]->removeReference();

	m_animation[1]->removeReference();
	hkaMirroredAnimation::destroyMirroredBinding( m_binding[1] );

	for( unsigned int i = 0; i < HK_COUNT_OF(m_skinBindings); ++i )
	{
		m_skinBindings[i].clear();
		m_attachments[i].clear();
	}

	delete m_loader;
}

void MirroredMotionDemo::makeFakeInput()
{
	// Fake a button press when demo is done easing
	{
		if (( m_control[0]->getEaseStatus() == hkaDefaultAnimationControl::EASED_OUT ) ||
			( m_control[0]->getEaseStatus() == hkaDefaultAnimationControl::EASED_IN ))
		{
			m_env->m_gamePad->forceButtonPressed(HKG_PAD_BUTTON_1);
		}
	}
	
}

hkDemo::Result MirroredMotionDemo::stepDemo()
{
	// Check user input
	{
		if (m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ))
		{
			for ( int i = 0; i < 2; i++ )
			{
				if (( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASING_IN ) ||
					( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASED_IN ))
				{
					m_control[i]->easeOut( .5f );
				}
				else
				{
					m_control[i]->easeIn( .5f );
				}
			}
		}

		if (m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ))
		{
			m_wireframe = !m_wireframe;
			setGraphicsState( HKG_ENABLED_WIREFRAME, m_wireframe );
		}

		if (m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ))
		{
			m_drawSkin = !m_drawSkin;
		}

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_0 ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_L1 ) )
		{
			m_accumulatedMotion[0].setIdentity();
			m_accumulatedMotion[1].setIdentity();
		}

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_R1 ) )
		{
			m_paused = !m_paused;

			if ( m_paused )
			{
				m_timestep = 0;
			}
			else
			{
				m_timestep = 1.0f / 60.0f;
			}
		}


	}

	for ( int j = 0; j < 2; j++ )
	{
		// Grab accumulated motion
		{
			hkQsTransform deltaMotion;
			deltaMotion.setIdentity();
			m_skeletonInstance[j]->getDeltaReferenceFrame( m_timestep, deltaMotion );
			m_accumulatedMotion[j].setMulEq( deltaMotion );
		}

		// Advance the active animations
		m_skeletonInstance[j]->stepDeltaTime( m_timestep );

		// Sample the active animations and combine into a single pose
		hkaPose pose (m_skeleton);
		m_skeletonInstance[j]->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), pose.getFloatSlotValues().begin() );

		hkReal separation = m_separation * hkReal( 2*j-1 );

		hkQsTransform Q( hkQsTransform::IDENTITY );
		Q.m_translation.set( separation, 0, 0 );

		if ( m_useExtractedMotion )
		{
			Q.setMulEq( m_accumulatedMotion[j] );
		}

		pose.syncModelSpace();

		AnimationUtils::drawPose( pose, Q );
		// Test if the skin is to be drawn
		if ( !m_drawSkin )
		{
			Q.m_translation( 2 ) -= 2.0f * m_separation;
		}

		hkTransform graphicsTransform( Q.m_rotation, hkVector4( Q.m_translation(0), Q.m_translation(1), Q.m_translation(2), 1 ) );
		AnimationUtils::skinMesh( pose, m_skinBindings[j], graphicsTransform, *m_env->m_sceneConverter );

		// Move the attachments
		{
			for (int a=0; a < m_attachments[j].getSize(); a++)
			{
				if (m_attachmentObjects[j][a])
				{
					hkaBoneAttachment* ba = m_attachments[j][a];
					hkQsTransform modelFromBone = pose.getBoneModelSpace( ba->m_boneIndex );
					hkQsTransform worldFromBoneQs;
					worldFromBoneQs.setMul( Q, modelFromBone );
					hkMatrix4 worldFromBone; worldFromBone.set(worldFromBoneQs);
					hkMatrix4 worldFromAttachment; worldFromAttachment.setMul(worldFromBone, ba->m_boneFromAttachment);
					m_env->m_sceneConverter->updateAttachment(m_attachmentObjects[j][a], worldFromAttachment);
				}
			}
		}


	}

	return hkDemo::DEMO_OK;
}




HK_DECLARE_DEMO_VARIANT_USING_STRUCT( MirroredMotionDemo, HK_DEMO_TYPE_OTHER, MirroredMotionVariant, g_variants, "Demo of mirroring animations" );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
