/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/MeshAndDeformation/HardwareSkinning/HardwareSkinningDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/System/hkgSystem.h> // to figure put if we should hardware skin
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Shader/hkgShaderLib.h>

HK_COMPILE_TIME_ASSERT(sizeof(hkaMeshBinding::Mapping) == sizeof(hkgAssetConverter::IndexMapping));

HardwareSkinningDemo::HardwareSkinningDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Disable warnings: if no renderer									
	if( hkString::strCmp( m_env->m_options->m_renderer, "n" ) == 0 )
	{
		setErrorEnabled(0xf0d1e423, false); //'Could not realize an inplace texture of type PNG.'
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(  -0.92f, -1.2f, 0.63f );
		hkVector4 to  ( 0.35f,-0.35f, 0.30f );
		hkVector4 up  ( 0.0f,  0.0f, 1.0f);
		setupDefaultCameras( env, from, to, up, 0.01f, 10000.0f );
	}

	m_loader = new hkLoader();

	// We want to use this demo to do the skinning, so we don't let our graphics engine do it.

	m_canSkin = (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_NULL); 
	if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_CONSOLE)
	{
		m_canSkin = HKG_GET_RENDER_CAPS(m_env->m_window).m_vertexShaderMajorVersion >= 2; // has 2.0 shaders..
	}
	env->m_sceneConverter->setAllowHardwareSkinning(m_canSkin);

	// Convert the scene
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene_Default.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

		HK_ASSERT2(0x27343635, scene, "No scene loaded");
		removeLights(m_env); // assume we have some in the file
		env->m_sceneConverter->convert( scene );
	}

	// Get the rig
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];
	}

	// Get the animation and the binding
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkIdle_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkIdle.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "No animation loaded");
		m_animation = ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding = ac->m_bindings[0];
	}

	// Convert the skin
	{
#if defined(HK_REAL_IS_DOUBLE)
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin18Bones_DP.hkt";
#else
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin18Bones.hkt";
#endif

		hkStringBuf assetFile(skinFile); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		env->m_sceneConverter->setShaderLibraryEnabled(true);
		if(m_canSkin)
		{
			env->m_sceneConverter->m_shaderLibrary->setAutoLoadShaderTypes(true, true); // load skin shaders in lib
		}

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );

		// If we are hardware skinning, we need to inform the graphics of the
		// mappings between the skeleton bones and the primitive subsets with
		// possibly limited palettes. Looks worse than it is as our scene converter
		// has no dependency on the animation system, so we need to cast a bit.
		if (m_canSkin)
		{
			for (int ms=0; ms < m_skinBindings.getSize(); ++ms)
			{
				hkaMeshBinding* skinBinding = m_skinBindings[ms];
				if ( !m_env->m_sceneConverter->setupHardwareSkin( m_env->m_window->getContext(), skinBinding->m_mesh,
						reinterpret_cast<hkgAssetConverter::IndexMapping*>( skinBinding->m_mappings.begin() ),
						skinBinding->m_mappings.getSize(), (hkInt16)skinBinding->m_skeleton->m_bones.getSize() ) )
				{
					HK_WARN_ALWAYS( 0x4327defe, "Could not setup the blend matrices for a skin. Will not be able to skin in h/w.");
				}
			}
		}
	}

	// Create the skeleton
	m_skeletonInstance = new hkaAnimatedSkeleton( m_skeleton );
	{
		hkaDefaultAnimationControl* ac = new hkaDefaultAnimationControl( m_binding );

		ac->setPlaybackSpeed(1.0f);
		m_skeletonInstance->addAnimationControl( ac );
		ac->removeReference();
	}

	// set up the vdb
	setupGraphics();

}

HardwareSkinningDemo::~HardwareSkinningDemo()
{
	m_skeletonInstance->removeReference();
	m_skinBindings.clear();

	delete m_loader;

	hkDefaultDemo::setupLights(m_env); // restore lights to default.

	// want to do software skinning by default in the demos
	m_env->m_sceneConverter->setAllowHardwareSkinning(false);
}

hkDemo::Result HardwareSkinningDemo::stepDemo()
{
	// Advance the active animations
	if (m_canSkin)
	{
		m_skeletonInstance->stepDeltaTime( .016f );

		hkaPose pose (m_skeleton);

		// Sample the active animations and combine into a single pose
		m_skeletonInstance->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), pose.getFloatSlotValues().begin()  );

		// Grab the pose in model space
		const hkArray<hkQsTransform>& poseModelSpace = pose.getSyncedPoseModelSpace();

		const int numSkins = m_skinBindings.getSize();
		for (int i=0; i < numSkins; i++)
		{
			const hkxMesh* inputMesh = m_skinBindings[i]->m_mesh;
			int numBones = m_skinBindings[i]->m_boneFromSkinMeshTransforms.getSize();

			// Compute the skinning matrix palette
			extArray<hkTransform> compositeWorldInverse( numBones );
			compositeWorldInverse.setSize( numBones );

			// Multiply through by the bind pose inverse world inverse matrices, according to the skel to mesh bone mapping
			for (int p=0; p < numBones; p++)
			{
				compositeWorldInverse[p].setMul( poseModelSpace[ p ], m_skinBindings[i]->m_boneFromSkinMeshTransforms[ p ] );
			}

			// reflect the matrices in the palette used by the shader
			m_env->m_sceneConverter->updateSkin( inputMesh, compositeWorldInverse, hkTransform::getIdentity() );
		}
	}

	return hkDemo::DEMO_OK;
}



HK_DECLARE_DEMO(HardwareSkinningDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "HardwareSkinning", "Play back a skinned mesh in available hardware.");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
