/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/LayeringAndPartitions/PartitionPlayback/PartitionPlaybackDemo.h>
#include <Animation/Animation/Animation/Interleaved/hkaInterleavedUncompressedAnimation.h>
#include <Animation/Animation/Animation/Quantized/hkaQuantizedAnimation.h>
#include <Animation/Animation/Animation/PredictiveCompressed/hkaPredictiveCompressedAnimation.h>
#include <Animation/Animation/Animation/SplineCompressed/hkaSplineCompressedAnimation.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Animation/Util/hkaPartitionedAnimationUtility.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

enum
{
	ANIMATION_TYPE_UNCOMPRESSED,
	ANIMATION_TYPE_SPLINE,
	ANIMATION_TYPE_QUANTIZED,
	ANIMATION_TYPE_PREDICTIVE,
};

hkaAnimation* PartitionPlaybackDemo::convertAnimation( const hkaAnimationBinding& binding, const hkaSkeleton& skeleton )
{
	HK_ASSERT( 0, binding.m_animation->getType() == hkaAnimation::HK_INTERLEAVED_ANIMATION );

	switch( m_variantId )
	{
	case ANIMATION_TYPE_SPLINE:
		{
			hkaInterleavedUncompressedAnimation* a = static_cast<hkaInterleavedUncompressedAnimation*>( binding.m_animation.val() );
			return new hkaSplineCompressedAnimation( *a );
		}
	case ANIMATION_TYPE_QUANTIZED:
		{
			return new hkaQuantizedAnimation( binding, skeleton );
		}
	case ANIMATION_TYPE_PREDICTIVE:
		{
			return new hkaPredictiveCompressedAnimation( binding, skeleton );
		}
	default:
		{
			hkaInterleavedUncompressedAnimation* a = static_cast<hkaInterleavedUncompressedAnimation*>( binding.m_animation.val() );
			a->addReference();
			return a;
		}
	}
}

void PartitionPlaybackDemo::createPartitionedAnimaton( hkaAnimationBinding& binding, const hkaSkeleton& skeleton, bool useFirstSetOfPartitions)
{
	hkaInterleavedUncompressedAnimation* origAnimation = static_cast<hkaInterleavedUncompressedAnimation*>( binding.m_animation.val() );

	hkArray<hkStringPtr> partitionNames;
	if(useFirstSetOfPartitions)
	{
		//Out Of Order
		partitionNames.pushBack("Chest-Head");
		partitionNames.pushBack("RightArm");
		partitionNames.pushBack("Legs");
	}
	else 
	{
		//In Order
		partitionNames.pushBack("Root");
		partitionNames.pushBack("Chest-Head");
		partitionNames.pushBack("LeftArm");
	}

	hkaPartitionedAnimationUtility::Input input;
	{
		input.m_animationData = origAnimation->m_transforms.begin();
		input.m_numberOfPoses = origAnimation->getNumOriginalFrames();
		input.m_numberOfTransformTracks = origAnimation->m_numberOfTransformTracks;
		input.m_skeleton = &skeleton;
		input.m_numberOfPartitions = partitionNames.getSize();
		input.m_partitionNames = partitionNames.begin();
		binding.m_partitionIndices.setSize(partitionNames.getSize());
	}

	//Output for the animation
	hkaPartitionedAnimationUtility::Output output;
	{
		output.m_transformsOut = origAnimation->m_transforms.begin();
		output.m_partitionIndicesOut = binding.m_partitionIndices.begin();
	}

	//Construct an animation with the partitions provided
	hkaPartitionedAnimationUtility::createPartitionedAnimation( input, output );

	origAnimation->m_transforms.setSize(output.m_numTotalTranformsOut);
	origAnimation->m_numberOfTransformTracks = output.m_numTransformTracksOut;

	//Partitioned animations require an identity mapping
	binding.m_transformTrackToBoneIndices.clear();
}

PartitionPlaybackDemo::PartitionPlaybackDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Disable warnings: if no renderer									
	if( hkString::strCmp( m_env->m_options->m_renderer, "n" ) == 0 )
	{
		setErrorEnabled(0xf0d1e423, false); //'Could not realize an inplace texture of type PNG.'
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from( 2,-2,1 );
		hkVector4 to  ( 0,0,0 );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	m_loader = new hkLoader();

	// Convert the scene
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene_Default.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

		HK_ASSERT2(0x27343635, scene, "No scene loaded");
		removeLights(m_env);
		env->m_sceneConverter->convert( scene );
	}

	// Get the rig
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];

		// Create partitions in the skeleton
		hkaSkeleton::Partition partition;
		{
			partition.m_name = "Root";
			partition.m_startBoneIndex = 0;
			partition.m_numBones = 3;
			m_skeleton->m_partitions.pushBack(partition);

			partition.m_name = "Chest-Head";
			partition.m_startBoneIndex = 3;
			partition.m_numBones = 7;
			m_skeleton->m_partitions.pushBack(partition);

			partition.m_name = "LeftArm";
			partition.m_startBoneIndex = 10;
			partition.m_numBones = 15;
			m_skeleton->m_partitions.pushBack(partition);

			partition.m_name = "RightArm";
			partition.m_startBoneIndex = 25;
			partition.m_numBones = 15;
			m_skeleton->m_partitions.pushBack(partition);

			partition.m_name = "Legs";
			partition.m_startBoneIndex = 40;
			partition.m_numBones = 10;
			m_skeleton->m_partitions.pushBack(partition);
		}
	}

	// Get the animation and the binding
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRunLoop_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRunLoop.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		createPartitionedAnimaton( *ac->m_bindings[0], *m_skeleton, true );
		m_animation[0] =  convertAnimation( *ac->m_bindings[0], *m_skeleton );

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[0] = ac->m_bindings[0];

#if defined(HK_REAL_IS_DOUBLE)
		assetFile = "Resources/Animation/HavokGirl/hkIdle_DP.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#else
		assetFile = "Resources/Animation/HavokGirl/hkIdle.hkt"; hkAssetManagementUtil::getFilePath(assetFile);
#endif
		container = m_loader->load( assetFile.cString() );
		ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		createPartitionedAnimaton( *ac->m_bindings[0], *m_skeleton, false );
		m_animation[1] =  convertAnimation( *ac->m_bindings[0], *m_skeleton );

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding[1] = ac->m_bindings[0];
	}

	// Convert the skin
	{
#if defined(HK_REAL_IS_DOUBLE)
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin_DP.hkt";
#else
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin.hkt";
#endif
		hkStringBuf assetFile(skinFile); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );
	}

	// Create the skeleton
	m_skeletonInstance = new hkaAnimatedSkeleton( m_skeleton );
	m_skeletonInstance->setReferencePoseWeightThreshold( 0.1f );

	// Grab the animations
	for (int i=0; i < NUM_ANIMS; i++)
	{
		m_control[i] = new hkaDefaultAnimationControl( m_binding[i] );
		m_control[i]->setMasterWeight( 1.0f );
		m_control[i]->setPlaybackSpeed( 1.0f );

		// Ease all controls out initially
		m_control[i]->easeOut(0.0f);

		m_skeletonInstance->addAnimationControl( m_control[i] );
		m_control[i]->removeReference();
	}

	// Set ease curves explicitly
	m_control[0]->setEaseInCurve(0, 0, 1, 1);	// Smooth
	m_control[0]->setEaseOutCurve(1, 1, 0, 0);	// Smooth

	m_control[1]->setEaseInCurve(0, 0.33f, 0.66f, 1);	// Linear
	m_control[1]->setEaseOutCurve(1, 0.66f, 0.33f, 0);	// Linear

	// make a world so that we can auto create a display world to hold the skin
	setupGraphics( );
}

PartitionPlaybackDemo::~PartitionPlaybackDemo()
{
	m_skeletonInstance->removeReference();
	m_animation[0]->removeReference();
	m_animation[1]->removeReference();
	m_skinBindings.clear();
	delete m_loader;
}

const hkColor::Argb curveCols[3] =
{
	hkColor::YELLOW,
	hkColor::MAGENTA,
	hkColor::CYAN
};

hkDemo::Result PartitionPlaybackDemo::stepDemo()
{

	// Display current settings
	{
		char buf[255];
		for (int i=0; i< NUM_ANIMS; i++)
		{
			hkString::sprintf(buf, "anim%d:%0.3f Partitions: %s, %s, %s", i, m_control[i]->getWeight(), 
								m_skeleton->m_partitions[m_binding[i]->m_partitionIndices[0]].m_name.cString(), 
								m_skeleton->m_partitions[m_binding[i]->m_partitionIndices[1]].m_name.cString(), 
								m_skeleton->m_partitions[m_binding[i]->m_partitionIndices[2]].m_name.cString());
			m_env->m_textDisplay->outputText( buf, 10, 240+14*i, curveCols[i], 1);
		}
	}

	// Check user input
	{
		for (int i=0; i < 2; i++)
		{
			if (m_env->m_gamePad->wasButtonPressed( 1 << (i+1) ))
			{
				if (( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASING_IN ) ||
					( m_control[i]->getEaseStatus() == hkaDefaultAnimationControl::EASED_IN ))
				{
					m_control[i]->easeOut( 1.0f );
				}
				else
				{
					m_control[i]->easeIn( 1.0f );
				}
			}
		}
	}

	// Advance the active animations
	m_skeletonInstance->stepDeltaTime( 1.0f / 60.0f );

	// Sample the active animations and combine into a single pose
	hkaPose pose (m_skeleton);
	pose.setToReferencePose();
	m_skeletonInstance->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), pose.getFloatSlotValues().begin() );
	AnimationUtils::drawPose( pose, hkQsTransform::getIdentity() );

	AnimationUtils::skinMesh( pose, m_skinBindings, hkTransform::getIdentity(), *m_env->m_sceneConverter );

	return hkDemo::DEMO_OK;
}



static const char* helpString = "Press \x11,\x12 to ease in / out different animations.";
HK_DECLARE_DEMO_VARIANT(PartitionPlaybackDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Uncompressed Partitions", ANIMATION_TYPE_UNCOMPRESSED, helpString, "Playback of an uncompressed animation with partitions");
HK_DECLARE_DEMO_VARIANT(PartitionPlaybackDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Spline Compression Partitions", ANIMATION_TYPE_SPLINE, helpString, "Playback of a spline compressed animation with partitions");
HK_DECLARE_DEMO_VARIANT(PartitionPlaybackDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Quantized Compression Partitions", ANIMATION_TYPE_QUANTIZED, helpString, "Playback of a quantized compressed animation with partitions");
HK_DECLARE_DEMO_VARIANT(PartitionPlaybackDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Predictive Compression Partitions", ANIMATION_TYPE_PREDICTIVE, helpString, "Playback of a predictive compressed animation with partitions");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
