/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/LayeringAndPartitions/Multithreading/RetargetingWithChains/RetargetingPartitionsWithChainsMT.h>
#include <Animation/Animation/Animation/Util/hkaPartitionedAnimationUtility.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Mapper/Multithreaded/hkaAnimationMappingJobQueueUtils.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJob.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJobQueueUtils.h>
#include <Animation/Animation/Playback/Multithreaded/hkaMultithreadedAnimationUtils.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/WindowedAverage/WindowedAverage.h>
#include <Animation/Animation/Rig/hkaSkeletonUtils.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Common/Base/Types/Color/hkColor.h>


static const int NUM_CHARACTERS = 10;

struct RetargetingWithChainsMultithreadingVariant
{
	const char*	m_name;
	const char* m_details;
	const char* m_rigFileNameA;
	const char* m_rigFileNameB;
	const char* m_animationFileName;
	const char* m_mappingFile;
	hkReal m_scale;
	hkBool m_densePose;
};

static const RetargetingWithChainsMultithreadingVariant g_variants[] =
{
	{
		"Animation Retargeting With Chains And Partitions Dense Sampling",
			"Press \x11 to toggle multithreaded mode",
			"Resources/Animation/Retargeting/singer_partitions.hkt",
			"Resources/Animation/Retargeting/gingerbread_partitions.hkt",
			"Resources/Animation/Retargeting/singer_partitions_sing_arms.hkt",
			"Resources/Animation/Retargeting/chain_mapping_partitions.hkt",
			20.0f,
			true
	},

	{
		"Animation Retargeting With Chains And Partitions Sparse Sampling",
			"Press \x11 to toggle multithreaded mode",
			"Resources/Animation/Retargeting/singer_partitions.hkt",
			"Resources/Animation/Retargeting/gingerbread_partitions.hkt",
			"Resources/Animation/Retargeting/singer_partitions_sing_arms.hkt",
			"Resources/Animation/Retargeting/chain_mapping_partitions.hkt",
			20.0f,
			false
		}
};

RetargetingPartitionsWithChainsMTDemo::RetargetingPartitionsWithChainsMTDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	m_useMt = false;
#else
	m_useMt = false;
#endif

	const RetargetingWithChainsMultithreadingVariant &variant = g_variants[ m_variantId ];

	m_loader = new hkLoader();

	//
	// Setup the camera
	//
	{
		hkVector4 from(  0.0f, -variant.m_scale,  0.0f);
		hkVector4 to  (  0.0f,  0.0f,   0.0f);
		hkVector4 up  (  0.0f,  0.0f,   1.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Skeletons
	{

		{
			hkStringBuf assetFile( variant.m_rigFileNameA );
			hkAssetManagementUtil::getFilePath( assetFile );
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

			HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load high-res skeleton");
			m_skeletonA = ac->m_skeletons[0];
		}

		{
			hkStringBuf assetFile( variant.m_rigFileNameB );
			hkAssetManagementUtil::getFilePath( assetFile );
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

			HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load low-res skeleton");
			m_skeletonB = ac->m_skeletons[0];
		}
	}

	// Animations
	{
		// High res animation
		hkStringBuf assetFile( variant.m_animationFileName );
		hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "Couldn't load high-res animation");
		m_animationForSkelA = ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "Couldn't load high-res binding");
		m_bindingForSkelA = ac->m_bindings[0];

		// Create several animated skeletons
		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			// Create an animation control
			hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_bindingForSkelA );

			// Create a new animated skeleton
			hkaAnimatedSkeleton* skeleton = new hkaAnimatedSkeleton( m_skeletonA );
			skeleton->setReferencePoseWeightThreshold(0.1f);

			// Bind the control to the skeleton
			skeleton->addAnimationControl( control );

			// The animated skeleton now owns the control
			control->removeReference();

			// Offset in time
			skeleton->stepDeltaTime( i * 0.25f );

			m_animatedSkeletonsForSkelA.pushBack( skeleton );
		}
	}

	// Poses
	{
		m_skelAPoses.setSize( NUM_CHARACTERS );
		m_skelBPoses.setSize( NUM_CHARACTERS );

		for ( int i = 0; i < m_animatedSkeletonsForSkelA.getSize(); ++i )
		{
			m_skelAPoses[ i ] = new hkaPose( m_skeletonA );
			m_skelBPoses[ i ] = new hkaPose( m_skeletonB );
		}
	}

	setupGraphics( );

#ifdef HK_PLATFORM_MULTI_THREAD
	hkaAnimationMappingJobQueueUtils::registerWithJobQueue( m_jobQueue );
	hkaSampleBlendJobQueueUtils::registerWithJobQueue( m_jobQueue );
#endif


	// Load the mapper
	{
		hkStringBuf assetFile( variant.m_mappingFile );
		hkAssetManagementUtil::getFilePath( assetFile );

		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

		m_skelAToSkelBMapper = reinterpret_cast< hkaSkeletonMapper* >( container->findObjectByType( hkaSkeletonMapperClass.getName() ) );
		HK_ASSERT2( 0x1d332f6d, m_skelAToSkelBMapper != HK_NULL, "No mapper found" );
	}

	//Setup the mapped partition indices for easy mapping later
	{
		for(int i = 0; i < m_bindingForSkelA->m_partitionIndices.getSize(); i++)
		{
			m_mappedBPartitionIndices.pushBack(m_skelAToSkelBMapper->m_mapping.m_partitionMap[m_bindingForSkelA->m_partitionIndices[i]]);
		}
	}
}


RetargetingPartitionsWithChainsMTDemo::~RetargetingPartitionsWithChainsMTDemo()
{
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		m_animatedSkeletonsForSkelA[ i ]->removeReference();

		delete m_skelAPoses[ i ];
		delete m_skelBPoses[ i ];
	}

	m_skelAToSkelBMapper->removeReference();
	delete m_loader;
}

hkDemo::Result RetargetingPartitionsWithChainsMTDemo::stepDemo()
{
	const RetargetingWithChainsMultithreadingVariant &variant = g_variants[ m_variantId ];

	const int n = NUM_CHARACTERS;

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) )
	{
		m_useMt = !m_useMt;
	}
#endif

	// Advance the animation of each of the hi-res skeletons
	for ( int i = 0; i < m_animatedSkeletonsForSkelA.getSize(); ++i )
	{
		m_animatedSkeletonsForSkelA[ i ]->stepDeltaTime( m_timestep );
	}

	// Fill the high resolution poses
	if ( m_useMt )
	{
		if(variant.m_densePose)
		{
			doSamplingAndMappingDenseMultithreaded();
		}
		else
		{
			// Sample multithreaded and wait for completion
			doSamplingMultithreaded();

			// Map multithreaded and wait for completion
			doMappingMultithreaded();
		}
	}
	else if(variant.m_densePose)
	{
		hkArray< hkArray<hkQsTransform> > tempPosesA(NUM_CHARACTERS); tempPosesA.setSize(NUM_CHARACTERS);
		hkArray< hkArray<hkQsTransform> > tempPosesB(NUM_CHARACTERS); tempPosesB.setSize(NUM_CHARACTERS);
		int numBonesInPartitionsA = m_bindingForSkelA->m_animation->m_numberOfTransformTracks;
		int numBonesInPartitionsB = hkaPartitionedAnimationUtility::getNumMappedPartitionBones(	m_skelAToSkelBMapper->m_mapping.m_partitionMap.begin(),
																							m_bindingForSkelA->m_partitionIndices.begin(),
																							m_bindingForSkelA->m_partitionIndices.getSize(),
																							m_skeletonB);

		for(int i = 0; i < NUM_CHARACTERS; i++)
		{
			tempPosesA[i].setSize(numBonesInPartitionsA);
			tempPosesB[i].setSize(numBonesInPartitionsB);
		}
		// Sample the high res animations
		for ( int i = 0; i < n; ++i )
		{
			//Sample the animation directly to avoid sample and combine expanding to a full pose
			m_animatedSkeletonsForSkelA[i]->getAnimationControl(0)->samplePartialTracks(numBonesInPartitionsA, tempPosesA[i].begin(), 0, HK_NULL);
		}

		// Map to low resolution
		for ( int i = 0; i < n; ++i )
		{
			//Both are going to be partial animatons so set them to the reference pose
			m_skelBPoses[ i ]->setToReferencePose();
			m_skelAPoses[ i ]->setToReferencePose();

			// The temp pose will be sparse so we need to set it the reference pose
			hkaPartitionedAnimationUtility::setPartitionPoseToReference(m_skeletonB,
																	m_mappedBPartitionIndices.begin(),
																	m_mappedBPartitionIndices.getSize(),
																	tempPosesB[i].begin());

			// Map the high-res pos into the low-res pose
			m_skelAToSkelBMapper->mapPoseLocalSpace(	tempPosesA[i].begin(),
														tempPosesB[i].begin(),
														m_bindingForSkelA->m_partitionIndices,
														hkaAnimationBinding::NORMAL,
														false);

			//map Skeleton A
			hkaPartitionedAnimationUtility::mapPartitionPoseToFullPose(	m_skeletonA->m_partitions,
																	m_bindingForSkelA->m_partitionIndices,
																	tempPosesA[i].begin(),
																	numBonesInPartitionsA,
																	m_skelAPoses[i]->accessUnsyncedPoseLocalSpace().begin());

			//map Skeleton B
			hkaPartitionedAnimationUtility::mapPartitionPoseToFullPose(	m_skeletonB->m_partitions,
																	m_mappedBPartitionIndices,
																	tempPosesB[i].begin(),
																	numBonesInPartitionsB,
																	m_skelBPoses[i]->accessUnsyncedPoseLocalSpace().begin());
		}
	}
	else
	{
		// Sample the high res animations
		for ( int i = 0; i < n; ++i )
		{
			m_animatedSkeletonsForSkelA[ i ]->sampleAndCombineAnimations( m_skelAPoses[ i ]->accessUnsyncedPoseLocalSpace().begin(), m_skelAPoses[ i ]->getFloatSlotValues().begin() );
		}

		// Map to low resolution
		for ( int i = 0; i < n; ++i )
		{
			m_skelBPoses[ i ]->setToReferencePose();

			// Map the high-res pos into the low-res pose
			m_skelAToSkelBMapper->mapPoseLocalSpace(	m_skelAPoses[ i ]->accessUnsyncedPoseLocalSpace().begin(),
				m_skelBPoses[ i ]->accessUnsyncedPoseLocalSpace().begin(),
				m_bindingForSkelA->m_partitionIndices,
				hkaAnimationBinding::NORMAL,
				true);
		}
	}

	// Draw both poses for each character
	{
		hkQsTransform worldFromModel( hkQsTransform::IDENTITY );

		// Draw the hi-res original poses
		for ( int i = 0; i < n; ++i )
		{
			worldFromModel.m_translation.set( ( hkReal( i ) / hkReal( n - 1 ) - 0.5f ) * variant.m_scale, 0, variant.m_scale / 5.0f );
			AnimationUtils::drawPose( *m_skelAPoses[ i ], worldFromModel, hkColor::GREEN );
		}

		// Draw the low-res mapped poses
		for ( int i = 0; i < n; ++i )
		{
			worldFromModel.m_translation.set( ( hkReal( i ) / hkReal( n - 1 ) -0.5f ) * variant.m_scale, 0, -variant.m_scale / 5.0f );
			AnimationUtils::drawPose( *m_skelBPoses[ i ], worldFromModel, hkColor::BLUE );
		}
	}

	// Display info
	{
		hkStringBuf str;
		str += "Animation Retargeting\n";
		str += m_useMt ? "Multi threaded" : "Single threaded";
		m_env->m_textDisplay->outputText( str.cString(), 20, (int)getWindowHeight() - 100 );
	}

	return hkDefaultAnimationDemo::stepDemo();
}

void RetargetingPartitionsWithChainsMTDemo::doMappingMultithreaded( )
{
	// Set up the jobs
	hkLocalArray< hkaAnimationMapPoseJob > mappingJobs( NUM_CHARACTERS );
	mappingJobs.setSize( NUM_CHARACTERS );

	HK_TIMER_BEGIN( "MappingJobSetup", HK_NULL );

	// Make jobs
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		// It is important for both retargeting and ragdoll mapping that the B pose be initialized before mapping
		// as not all bones may be mapped
		m_skelBPoses[ i ]->setToReferencePose();

		// This semaphore is released when the job is complete (if it's non-null)
		mappingJobs[ i ].m_jobDoneSemaphore = HK_NULL;

		mappingJobs[ i ].m_mapper = m_skelAToSkelBMapper;
		mappingJobs[ i ].m_poseBLocalSpaceIn = m_skelBPoses[ i ]->getSyncedPoseLocalSpace().begin();
		mappingJobs[ i ].m_source = hkaSkeletonMapper::CURRENT_POSE;
		mappingJobs[ i ].m_blendHint = hkaAnimationBinding::NORMAL;

		mappingJobs[ i ].m_poseAIn = m_skelAPoses[ i ]->getSyncedPoseLocalSpace().begin();
		mappingJobs[ i ].m_poseBInOut = m_skelBPoses[ i ]->accessSyncedPoseLocalSpace().begin();

		mappingJobs[ i ].m_partitionIndicesA = m_bindingForSkelA->m_partitionIndices.begin();
		mappingJobs[ i ].m_numPartitionsIndices = (hkInt16) m_bindingForSkelA->m_partitionIndices.getSize();
		mappingJobs[ i ].m_mapToFullPose = true;

		// Add the job - execution begins immediately on the SPU.
		// Since however this uses a critical section, and the job set up code above is relatively fast,
		// we defer adding until all jobs are set up, and then use an addJobBatch - this will be faster.
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN( "MappingAddJobBatch", HK_NULL );
		hkLocalArray<hkJob*> jobPointers( NUM_CHARACTERS );
		jobPointers.setSize( NUM_CHARACTERS );

		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			jobPointers[i] = &( mappingJobs[ i ] );
		}

		// Add the job - execution begins immediately on the SPU
		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	HK_TIMER_END();

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();
}


void RetargetingPartitionsWithChainsMTDemo::doSamplingMultithreaded()
{
	const int numSkeletons = m_animatedSkeletonsForSkelA.getSize();
	// Set up the jobs
	hkLocalArray<hkaSampleBlendJob> sampleBlendJobs(numSkeletons);
	sampleBlendJobs.setSize(numSkeletons);

	// Make jobs
	for (int i = 0; i < numSkeletons; ++i )
	{
		HK_TIMER_BEGIN( "build job", HK_NULL );

		sampleBlendJobs[i].build(
			m_animatedSkeletonsForSkelA[i],
			m_skelAPoses[i]->accessUnsyncedPoseLocalSpace().begin(),
			HK_NULL);

		HK_TIMER_END();
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN( "AddJobBatch", HK_NULL );
		hkLocalArray<hkJob*> jobPointers( numSkeletons );
		jobPointers.setSize( numSkeletons );
		for ( int i = 0; i < numSkeletons; ++i )
		{
			jobPointers[i] = &( sampleBlendJobs[i] );
		}

		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	HK_TIMER_END();

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();

	// Clean up the control buffers
	for (int i = 0; i < numSkeletons; ++i )
	{
		sampleBlendJobs[i].destroy();
	}
}

void RetargetingPartitionsWithChainsMTDemo::doSamplingAndMappingDenseMultithreaded()
{
	int numThreads = m_threadPool->getNumThreads();
	numThreads++; // count the main thread too

	//For Sample Only we get back just a partial animation with the partition's transforms
	hkArray< hkArray<hkQsTransform> > tempPosesA(NUM_CHARACTERS); tempPosesA.setSize(NUM_CHARACTERS);
	hkArray< hkArray<hkQsTransform> > tempPosesB(NUM_CHARACTERS); tempPosesB.setSize(NUM_CHARACTERS);
	int numBonesInPartitionsA = m_bindingForSkelA->m_animation->m_numberOfTransformTracks;
	int numBonesInPartitionsB = hkaPartitionedAnimationUtility::getNumMappedPartitionBones(	m_skelAToSkelBMapper->m_mapping.m_partitionMap.begin(),
																						m_bindingForSkelA->m_partitionIndices.begin(),
																						m_bindingForSkelA->m_partitionIndices.getSize(),
																						m_skeletonB);

	for(int i = 0; i < NUM_CHARACTERS; i++)
	{
		tempPosesA[i].setSize(numBonesInPartitionsA);
		tempPosesB[i].setSize(numBonesInPartitionsB);
	}

	// Set up the jobs
	hkLocalArray<hkaSampleBlendJob> sampleJobs(NUM_CHARACTERS);
	sampleJobs.setSize(NUM_CHARACTERS);

	int charIndex = 0;

	// Make jobs
	for (int s=0; s < numThreads; s++)
	{
		const int animsForJob = NUM_CHARACTERS / numThreads + int(s < NUM_CHARACTERS % numThreads);	// handles any remainders

		sampleJobs[s].initSampleOnly( animsForJob, *m_skeletonA, false, numBonesInPartitionsA );

		for( int i = 0; i < animsForJob; i++ )
		{
			hkReal localTime = m_animatedSkeletonsForSkelA[charIndex]->getAnimationControl(0)->getLocalTime();
			hkQsTransform* bonesBuffer = tempPosesA[charIndex].begin();
			sampleJobs[s].addAnimation( *m_skeletonA, *m_bindingForSkelA->m_animation, localTime, bonesBuffer, HK_NULL, m_bindingForSkelA );
			charIndex++;
		}
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN("AddJobBatch", HK_NULL);
		hkLocalArray<hkJob*> jobPointers(numThreads);
		jobPointers.setSize(numThreads);
		for (int i = 0; i < numThreads; ++i )
		{
			jobPointers[i] = &( sampleJobs[i] );
		}
		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();

	// Clean up the control buffers
	for (int i = 0; i < NUM_CHARACTERS; ++i )
	{
		sampleJobs[i].destroy();
	}

	// Set up the jobs
	hkLocalArray< hkaAnimationMapPoseJob > mappingJobs( NUM_CHARACTERS );
	mappingJobs.setSize( NUM_CHARACTERS );

	HK_TIMER_BEGIN( "MappingJobSetup", HK_NULL );

	// Make jobs
	for ( int i = 0; i < NUM_CHARACTERS; ++i )
	{
		// It is important for both retargeting and ragdoll mapping that the B pose be initialized before mapping
		// as not all bones may be mapped
		// The temp pose will be sparse so we need to set it the reference pose
		hkaPartitionedAnimationUtility::setPartitionPoseToReference(m_skeletonB,
			m_bindingForSkelA->m_partitionIndices.begin(),
			m_bindingForSkelA->m_partitionIndices.getSize(),
			tempPosesB[i].begin());

		// This semaphore is released when the job is complete (if it's non-null)
		mappingJobs[ i ].m_jobDoneSemaphore = HK_NULL;

		mappingJobs[ i ].m_mapper = m_skelAToSkelBMapper;
		mappingJobs[ i ].m_poseBLocalSpaceIn = tempPosesB[ i ].begin();
		mappingJobs[ i ].m_source = hkaSkeletonMapper::CURRENT_POSE;
		mappingJobs[ i ].m_blendHint = hkaAnimationBinding::NORMAL;

		mappingJobs[ i ].m_poseAIn = tempPosesA[ i ].begin();
		mappingJobs[ i ].m_poseBInOut = tempPosesB[ i ].begin();

		mappingJobs[ i ].m_numPartitionTransformsA = (hkInt16)numBonesInPartitionsA;
		mappingJobs[ i ].m_numPartitionTransformsB = (hkInt16)numBonesInPartitionsB;

		mappingJobs[ i ].m_partitionIndicesA = m_bindingForSkelA->m_partitionIndices.begin();
		mappingJobs[ i ].m_numPartitionsIndices = (hkInt16) m_bindingForSkelA->m_partitionIndices.getSize();
		mappingJobs[ i ].m_mapToFullPose = false;

		// Add the job - execution begins immediately on the SPU.
		// Since however this uses a critical section, and the job set up code above is relatively fast,
		// we defer adding until all jobs are set up, and then use an addJobBatch - this will be faster.
	}

	// Add jobs as a batch
	{
		HK_TIMER_BEGIN( "MappingAddJobBatch", HK_NULL );
		hkLocalArray<hkJob*> jobPointers( NUM_CHARACTERS );
		jobPointers.setSize( NUM_CHARACTERS );

		for ( int i = 0; i < NUM_CHARACTERS; ++i )
		{
			jobPointers[i] = &( mappingJobs[ i ] );
		}

		// Add the job - execution begins immediately on the SPU
		m_jobQueue->addJobBatch( jobPointers, hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_END();
	}

	HK_TIMER_END();

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for all threads to finish

	// There's no need to wait on the hkaAnimationSampleAndCombineJob's semaphore here, since we're going to end up waiting for all the jobs to finish.
	// However, if each job had its own semaphore and we wanted to wait on an specific job, this would be the place to do it.

	// Wait for the actual tasks to finish. This makes sure all timer information will have finished DMAing to main memory
	m_threadPool->waitForCompletion();

	for(int i = 0; i < NUM_CHARACTERS; i++)
	{
		m_skelBPoses[ i ]->setToReferencePose();
		m_skelAPoses[ i ]->setToReferencePose();

		//map Skeleton A
		hkaPartitionedAnimationUtility::mapPartitionPoseToFullPose(	m_skeletonA->m_partitions,
																m_bindingForSkelA->m_partitionIndices,
																tempPosesA[i].begin(),
																numBonesInPartitionsA,
																m_skelAPoses[i]->accessUnsyncedPoseLocalSpace().begin());

		//map Skeleton B
		hkaPartitionedAnimationUtility::mapPartitionPoseToFullPose(	m_skeletonB->m_partitions,
																m_bindingForSkelA->m_partitionIndices, //For this demo the indidces are the same
																tempPosesB[i].begin(),
																numBonesInPartitionsB,
																m_skelBPoses[i]->accessUnsyncedPoseLocalSpace().begin());
	}
}





static const char* description = "Demo showing animation skeleton mapping with chains and partitions using multithreaded jobs";

#if defined(HK_REAL_IS_DOUBLE)
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( RetargetingPartitionsWithChainsMTDemo, HK_DEMO_TYPE_ANIMATION , RetargetingWithChainsMultithreadingVariant, g_variants, description );
#else
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( RetargetingPartitionsWithChainsMTDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL , RetargetingWithChainsMultithreadingVariant, g_variants, description );
#endif

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
