/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/InverseKinematics/Reaching/ReachingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Ik/Ccd/hkaCcdIkSolver.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/Rig/hkaSkeletonUtils.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

ReachingDemo::ReachingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Disable warnings: if no renderer									
	if( hkString::strCmp( m_env->m_options->m_renderer, "n" ) == 0 )
	{
		setErrorEnabled(0xf0d1e423, false); //'Could not realize an inplace texture of type PNG.'
	}
	
	//
	// Setup the camera
	//
	{
		hkVector4 from( -2,-2,2 );
		hkVector4 to  ( 0,0,0 );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	m_loader = new hkLoader();

	// Convert the scene
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene_Default.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/Scene/hkScene.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

		HK_ASSERT2(0x27343635, scene, "No scene loaded");
		removeLights(m_env);
		env->m_sceneConverter->convert( scene );
	}

	// Get the rig
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];
	}

	// Get the idle animation and the binding
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkIdle_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkIdle.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		m_animation =  ac->m_animations[0];

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		m_binding = ac->m_bindings[0];
	}

	// Convert the skin
	{
#if defined(HK_REAL_IS_DOUBLE)
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin_DP.hkt";
#else
		const char* skinFile = "Resources/Animation/HavokGirl/hkLowResSkin.hkt";
#endif
		hkStringBuf assetFile(skinFile); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );
	}

	// Create the skeleton
	m_skeletonInstance = new hkaAnimatedSkeleton( m_skeleton );

	// Set the fill threshold
	m_skeletonInstance->setReferencePoseWeightThreshold( 0.05f );

	// animation controller
	{
		m_control = new hkaDefaultAnimationControl( m_binding );
		m_control->setPlaybackSpeed( 1.0f );

		m_skeletonInstance->addAnimationControl( m_control );
		m_control->removeReference();
	}

	// setup the graphics
	setupGraphics( );

	// find the bone indices for the left hand index finger and the left clavicle
	m_lFingerIndex = hkaSkeletonUtils::findBoneWithName( *m_skeleton, "HavokBipedRig L Hand" );
	m_lClavicleIndex = hkaSkeletonUtils::findBoneWithName( *m_skeleton, "HavokBipedRig L Clavicle" );

	// set the initial end position for the left finger
	m_targetPosition.set( 0.3f, -0.2f, 0.1f );

}


ReachingDemo::~ReachingDemo()
{
	m_skeletonInstance->removeReference();

	m_skinBindings.clear();

	delete m_loader;

	hkDefaultDemo::setupLights(m_env); // assumes the hkx will have some.
}


hkDemo::Result ReachingDemo::stepDemo()
{
	const hkReal inc = 0.01f;

	// check for any input changes
	{
		const hkgPad* pad = m_env->m_gamePad;
		if ((pad->getButtonState() & HKG_PAD_DPAD_UP) != 0)
		{
			m_targetPosition(2) += inc;
		}

		if ((pad->getButtonState() & HKG_PAD_DPAD_LEFT) != 0)
		{
			m_targetPosition(0) -= inc;
		}

		if ((pad->getButtonState() & HKG_PAD_DPAD_RIGHT) != 0)
		{
			m_targetPosition(0) += inc;
		}

		if ((pad->getButtonState() & HKG_PAD_DPAD_DOWN) != 0)
		{
			m_targetPosition(2) -= inc;
		}
	}

	// display a blue star at the target finger position
	HK_DISPLAY_STAR( m_targetPosition, 0.15f, 0x000000ff );

	// Advance the active animations
	m_skeletonInstance->stepDeltaTime( 0.016f );

	hkaPose thePose (m_skeleton);
	// Sample the active animations and combine into a single pose
	m_skeletonInstance->sampleAndCombineAnimations( thePose.accessUnsyncedPoseLocalSpace().begin(), thePose.getFloatSlotValues().begin()  );

	// Solve the IK
	{
		// an array of constraints to be solved
		hkLocalArray<hkaCcdIkSolver::IkConstraint> constraints(1);
		constraints.setSize(1);

		// set the IK bone parameters
		constraints[0].m_endBone = m_lFingerIndex;
		constraints[0].m_startBone = m_lClavicleIndex;
		// set the IK target paramter
		constraints[0].m_targetMS = m_targetPosition;

		// perform the CCD IK
		hkaCcdIkSolver ccdIkSolver;
		ccdIkSolver.solve( constraints, thePose );
	}

	// get the bone indices for the fore arm and twist bones ( to allow re-sync'ing after IK )
	// The twist bones are procedurally linked to the lower arm in Character studio.
	hkInt16 twist = hkaSkeletonUtils::findBoneWithName( *m_skeleton, "HavokBipedRig L ForeTwist" );
	hkInt16 foreArm = hkaSkeletonUtils::findBoneWithName( *m_skeleton, "HavokBipedRig L ForeArm" );

	// re-sync the twist bone with the forearm
	thePose.setBoneLocalSpace(twist, thePose.getBoneLocalSpace(foreArm));

	// draw the skeleton pose
	AnimationUtils::drawPose( thePose, hkQsTransform::getIdentity() );

	// Skin
	AnimationUtils::skinMesh( thePose, m_skinBindings, hkTransform::getIdentity(), *m_env->m_sceneConverter );

	return hkDemo::DEMO_OK;
}


static const char helpString[] = \
"A simple IK example that moves a character's hand to a given position.\n" \
"The target position may be changed by pressing up, down, left and right.";

HK_DECLARE_DEMO(ReachingDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Character reaching for a given target", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
