/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Compression/Advanced/PerBoneCompressionDemo.h>
#include <Animation/Animation/Animation/Interleaved/hkaInterleavedUncompressedAnimation.h>
#include <Animation/Animation/Animation/Quantized/hkaQuantizedAnimation.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

static hkVector4 OFFSET( 0.3f, 0, 0, 0 );

class TestRecord
{
public:

	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, TestRecord);

	hkaAnimationBinding* m_binding;
	hkaAnimatedSkeleton* m_animatedSkeleton;
	const char* m_name;
	int m_bytes;
	hkVector4 m_offset;
	hkStopwatch m_stopwatch;
};

static int distanceFromRoot(hkInt16 boneId, const hkaSkeleton& skeleton )
{
	const hkInt16 parent = skeleton.m_parentIndices[boneId];
	return (parent == -1) ? 0 : distanceFromRoot( parent, skeleton ) + 1;
}

void PerBoneCompressionDemo::AddAnimation(const char* name, hkaAnimation* anim, hkaAnimationBinding* binding )
{
	TestRecord* rec = new TestRecord;

	rec->m_name = name;
	rec->m_bytes = 0;

	// Set up the binding
	rec->m_binding = new hkaAnimationBinding(*binding);
	rec->m_binding->m_animation = anim;

	// Create an animation control
	hkaDefaultAnimationControl* ac = new hkaDefaultAnimationControl(rec->m_binding);

	// Create a new animated skeleton
	rec->m_animatedSkeleton = new hkaAnimatedSkeleton( m_skeleton );
	rec->m_animatedSkeleton->setReferencePoseWeightThreshold(0.001f);

	// Bind the control to the skeleton
	rec->m_animatedSkeleton->addAnimationControl( ac );

	// The animated skeleton now owns the control
	ac->removeReference();

	rec->m_offset = OFFSET;
	rec->m_offset.mul4( hkReal(m_animationRecords.getSize()) );

	m_animationRecords.pushBack(rec);
}



PerBoneCompressionDemo::PerBoneCompressionDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env), m_currentAnimation(0)
{
	
	// Disable reports: 									
	if( m_env->m_reportingLevel < hkDemoEnvironment::REPORT_INFO )
	{
		setErrorEnabled(0x432434a4, false); 
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from( 0, -6, 1);
		hkVector4 to  (  0.0f,  0.0f,   0.0f);
		hkVector4 up  (  0.0f,  0.0f,   1.0f);
		setupDefaultCameras( env, from, to, up );
	}

	m_loader = new hkLoader();

	// Get the rig
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];
	}

	// Get the animation and the binding
	hkaInterleavedUncompressedAnimation* rawAnimation;
	hkaAnimationBinding* binding;
	{
#if defined(HK_REAL_IS_DOUBLE)
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkWalkLoop_DP.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#else
		hkStringBuf assetFile("Resources/Animation/HavokGirl/hkWalkLoop.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
#endif
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "No animation loaded");
		rawAnimation = static_cast<hkaInterleavedUncompressedAnimation*>(ac->m_animations[0].val());

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		binding = ac->m_bindings[0];
	}

	// Raw
	AddAnimation("Original Raw", rawAnimation, binding);
	m_originalSize = rawAnimation->m_transforms.getSize() * sizeof(hkQsTransform);
	m_animationRecords.back()->m_bytes = m_originalSize;

	const int NUM_ANIMS = 2;
	const char* compressionDescriptions[NUM_ANIMS];
	hkaQuantizedAnimation::PerTrackCompressionParams perTrackParams[NUM_ANIMS];

	// all bones have the same tolerance
	{
		compressionDescriptions[0] = "All bones have tolerances of 0.0001";
		hkaQuantizedAnimation::PerTrackCompressionParams& cInfo = perTrackParams[0];

		hkaQuantizedAnimation::TrackCompressionParams& p = cInfo.m_parameterPalette.expandOne();

		p.m_rotationTolerance = 0.0001f;
		p.m_translationTolerance = 0.0001f;
		p.m_scaleTolerance = 0.0001f;
		p.m_floatingTolerance = 0.0001f;

		// Set all bones to use this setting
		cInfo.m_trackIndexToPaletteIndex.setSize( rawAnimation->m_numberOfTransformTracks, 0 );
	}

	// higher compression on leafward nodes
	{
		compressionDescriptions[1] = "Rootward bones have tolerances of 0.0001\nLeafward Bones have tolerances of 0.01";

		hkaQuantizedAnimation::PerTrackCompressionParams& cInfo = perTrackParams[1];

		cInfo.m_parameterPalette.setSize( 2, hkaQuantizedAnimation::TrackCompressionParams() );

		cInfo.m_parameterPalette[0].m_rotationTolerance = 0.0001f;
		cInfo.m_parameterPalette[0].m_translationTolerance = 0.0001f;
		cInfo.m_parameterPalette[0].m_scaleTolerance = 0.0001f;
		cInfo.m_parameterPalette[0].m_floatingTolerance = 0.0001f;

		cInfo.m_parameterPalette[1].m_rotationTolerance = 0.01f;
		cInfo.m_parameterPalette[1].m_translationTolerance = 0.01f;
		cInfo.m_parameterPalette[1].m_scaleTolerance = 0.01f;
		cInfo.m_parameterPalette[1].m_floatingTolerance = 0.01f;

		// Set up the palette for each track in the animation - ones further from the root have more compression.
		cInfo.m_trackIndexToPaletteIndex.setSize( rawAnimation->m_numberOfTransformTracks );
		for (int t=0; t < rawAnimation->m_numberOfTransformTracks; t++)
		{
			hkInt16 boneId = binding->m_transformTrackToBoneIndices[t];
			cInfo.m_trackIndexToPaletteIndex[t] = distanceFromRoot(boneId, *m_skeleton) < 6 ? 0 : 1;
		}
	}

	// compress the animations
	for (int w=0; w < NUM_ANIMS ; w++)
	{
		hkaQuantizedAnimation* anim =  new hkaQuantizedAnimation( *binding, *m_skeleton, perTrackParams[w] );
		AddAnimation(compressionDescriptions[w], anim, binding);
		m_animationRecords.back()->m_bytes = anim->getSizeInBytes();
	}

	setupGraphics( );

}

PerBoneCompressionDemo::~PerBoneCompressionDemo()
{
	// Delete the active skeletons
	for (int s=0; s< m_animationRecords.getSize(); s++)
	{
		if (s > 0)
		{
			m_animationRecords[s]->m_binding->m_animation->removeReference();
		}
		m_animationRecords[s]->m_binding->removeReference();
		m_animationRecords[s]->m_animatedSkeleton->removeReference();

		delete m_animationRecords[s];
	}

	delete m_loader;
}

hkDemo::Result PerBoneCompressionDemo::stepDemo()
{
	if ( m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1) )
	{
		m_currentAnimation = (m_currentAnimation+1)%(m_animationRecords.getSize());
		hkVector4 at = m_animationRecords[m_currentAnimation]->m_offset;
		lookAt( at );
	}

	hkaPose pose (m_skeleton);

	for (int i = 0; i < m_animationRecords.getSize(); ++i )
	{
		TestRecord* rec = m_animationRecords[i];
		hkaAnimatedSkeleton* inst = rec->m_animatedSkeleton;

		// Advance the animation
		rec->m_stopwatch.reset();
		rec->m_stopwatch.start();

		inst->stepDeltaTime( m_timestep );

		// Sample
		inst->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), pose.getFloatSlotValues().begin()  );

		rec->m_stopwatch.stop();

		// Draw
		hkQsTransform worldFromModel (hkQsTransform::IDENTITY);
		worldFromModel.m_translation = rec->m_offset;
		hkColor::Argb color = (i == m_currentAnimation) ? hkColor::RED : hkColor::WHITE;
		AnimationUtils::drawPose( pose, worldFromModel, color );
	}

	TestRecord* rec = m_animationRecords[m_currentAnimation];

	char msg[2048];

	hkUint64 ticks = rec->m_stopwatch.getElapsedTicks();
	double tpf = double(ticks/rec->m_stopwatch.getNumTimings());
	double tpms = double(rec->m_stopwatch.getTicksPerSecond()/1000000);
	double mspf = tpf / tpms;
	hkString::sprintf(msg, "%s\nbytes:%d\nratio:%.2f:1\nmicrosecs/f:%f", rec->m_name, rec->m_bytes, (hkReal)m_originalSize / rec->m_bytes, mspf );
	const int h = getWindowHeight();
	m_env->m_textDisplay->outputText( msg, 20, h-100, 0xffffffff);

	return DEMO_OK;
}

void PerBoneCompressionDemo::lookAt( const hkVector4& to )
{
	hkgWindow* w = m_env->m_window;
	hkVector4 offset, from, oldTo;

	for(int i = 0; i < w->getNumViewports(); ++i)
	{
		hkgCamera* c = w->getViewport(i)->getCamera();

		c->getFrom( from );
		c->getTo( oldTo );
		offset.setSub4(to, oldTo);
		from.add4(offset);

		c->setFrom(&from(0));
		c->setTo(&to(0));
		c->orthogonalize();
		c->computeModelView();
		c->computeProjection();
	}
}


static const char helpString[] = \
"This demo takes a single raw animation and compresses it with different "
"compression settings. Different compression settings are used for different bones in the hierarchy. "
"Bones further from the root (leafward) are compressed more. Press \x11 to switch and see results.";

HK_DECLARE_DEMO(PerBoneCompressionDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE | HK_DEMO_TYPE_CRITICAL, "Per track compression", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
